import time

import canopen
import can.interfaces
from zlgcanadapter import ZLGCanNetBus


class Canopen_ku:

    def __init__(self, bustype1, channel1, baud1=125000, node_ids=[]):
        self.bustype1 = bustype1
        self.channel1 = channel1
        self.baud1 = baud1
        if self.bustype1 == 'zlgcanet':
            zlgcannet_bus = ZLGCanNetBus(ip=channel1.split(":")[0], port=channel1.split(":")[1])
            self.network = canopen.Network()
            self.network.bus = zlgcannet_bus
            self.network.notifier = can.Notifier(zlgcannet_bus, self.network.listeners)
        else:
            self.network = canopen.Network()
            self.network.connect(bustype=self.bustype1, channel=self.channel1, bitrate=self.baud1)
        self.node_map = {}
        for node_id in node_ids:
            node = self.network.add_node(node_id, 'PMC007CX.EDS')
            self.node_map[node_id] = node

    # 获取节点ID
    def read_salve_id(self, node_id):
        value = self.node_map[node_id].sdo[0x2002].raw
        return value

    # 设置节点id
    def set_salve_id(self, node_id, value=5):
        self.node_map[node_id].sdo[0x2002].raw = value

    # 获取波特率
    def read_baudrate(self, node_id):
        value = self.node_map[node_id].sdo[0x2003].raw
        if value == 0:
            baud = 20000
        elif value == 1:
            baud = 25000
        elif value == 2:
            baud = 50000
        elif value == 3:
            baud = 100000
        elif value == 4:
            baud = 125000
        elif value == 5:
            baud = 250000
        elif value == 6:
            baud = 500000
        elif value == 7:
            baud = 800000
        elif value == 8:
            baud = 10000000
        else:
            baud = '输入错误'
        return baud

    # 设置波特率 0-20kbit   1-25kbit   2-50kbit   3-100kbit  4-125kbit
    #         5-250kbit  6-500kbit  7-800kbit  8-1000kbit
    def set_baudrate(self, node_id, value=5):
        if value == 0:
            self.node_map[node_id].sdo[0x2003].raw = 20000
        elif value == 1:
            self.node_map[node_id].sdo[0x2003].raw = 25000
        elif value == 2:
            self.node_map[node_id].sdo[0x2003].raw = 50000
        elif value == 3:
            self.node_map[node_id].sdo[0x2003].raw = 100000
        elif value == 4:
            self.node_map[node_id].sdo[0x2003].raw = 1250000
        elif value == 5:
            self.node_map[node_id].sdo[0x2003].raw = 250000
        elif value == 6:
            self.node_map[node_id].sdo[0x2003].raw = 500000
        elif value == 7:
            self.node_map[node_id].sdo[0x2003].raw = 800000
        elif value == 8:
            self.node_map[node_id].sdo[0x2003].raw = 10000000

    # 获取组id
    def read_zu_id(self, node_id):
        value = self.node_map[node_id].sdo[0x2006].raw
        return value

    # 设置组id
    def set_zu_id(self, node_id, value=0):
        self.node_map[node_id].sdo[0x2006].raw = value

    # 获取设备节点名称
    def read_device_node(self, node_id):
        value = self.node_map[node_id].sdo[0x1008].raw
        return value

    # 硬件版本
    def read_hardware_version(self, node_id):
        value = self.node_map[node_id].sdo[0x1009].raw
        return value

    # 软件版本
    def read_software_version(self, node_id):
        value = self.node_map[node_id].sdo[0x100A].raw
        return value

    # 读系统控制
    def read_system_control(self, node_id):
        value = self.node_map[node_id].sdo[0x2007].raw
        return value

    # 设置系统控制
    def set_system_control(self, node_id, word):
        self.node_map[node_id].sdo[0x2007].raw = word

    # 获取驱动状态
    def read_driving_conditions(self, node_id):
        value = self.node_map[node_id].sdo[0x6000].raw
        a = '{:08b}'.format(value)  # 将得到的十进制转化为八位二进制数,字符串类型
        return a

    # 设置驱动状态(bit位，bit0~5:TSD,AERR,BERR,AOC,BOC,UVLO)
    def set_driving_conditions(self, node_id, word=0):
        self.node_map[node_id].sdo[0x6000].raw = word

    # 获取控制器状态
    def read_controller_status(self, node_id):
        value = self.node_map[node_id].sdo[0x6001].raw
        return value

    # 设置控制器状态 bit位u8  bit0~3：外部停止1，外部停止2，堵转状态，busy状态
    # bit4~7：外部停止3，pvt模式3的FIFO为空，pvt模式3的FIFO下限，pvt模式3的FIFO上限
    def set_controller_status(self, node_id, word):
        self.node_map[node_id].sdo[0x6001].raw = word

    # 获取转动方向
    def read_rotation_direction(self, node_id):
        value = self.node_map[node_id].sdo[0x6002].raw
        return value

    # 设置转动方向 0正向，1反向
    def set_rotation_direction(self, node_id, word):
        self.node_map[node_id].sdo[0x6002].raw = word

    # 获取最大速度
    def read_max_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6003].raw
        return value

    # 设置最大速度 (s32 范围：-200000~+200000）
    def set_max_speed(self, node_id, word):
        self.node_map[node_id].sdo[0x6003].raw = word

    # 读相对位移指令（范围：0x0~0xFFFFFFFF）
    def read_step(self, node_id):
        value = self.node_map[node_id].sdo[0x6004].raw
        return value

    # 设置相对位移指令（u32 范围：0x0~0xFFFFFFFF）
    def set_step(self, node_id, word):
        self.node_map[node_id].sdo[0x6004].raw = word

    # 获取绝对位移指令
    def read_absolute_displacement(self, node_id):
        value = self.node_map[node_id].sdo[0x601c].raw
        return value

    # 设置绝对位移指令
    def set_absolute_displacement(self, node_id, value):
        self.node_map[node_id].sdo[0x601c].raw = value

    # 终止步进指令
    def stop_step(self, node_id, word):
        self.node_map[node_id].sdo[0x6020].raw = word

    # 读取工作模式
    def read_work_mode(self, node_id):
        value = self.node_map[node_id].sdo[0x6005].raw
        return value

    # 选择工作模式（0.位置模式；1.速度模式；2.pvt模式；3.编码器跟随模式；4.pp模式；5.pv模式）
    def set_work_mode(self, node_id, mode):
        self.node_map[node_id].sdo[0x6005].raw = mode

    # 获取启动速度
    def read_start_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6006].raw
        return value

    # 设置启动速度（范围 0-0xFFFF）
    def set_start_speed(self, node_id, speed):
        self.node_map[node_id].sdo[0x6006].raw = speed
    # 获取停止速度

    def read_stop_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6007].raw
        return value

    # 设置停止速度（范围 0-0xFFFF）
    def set_stop_speed(self, node_id, speed):
        self.node_map[node_id].sdo[0x6007].raw = speed

    # 获取加速度系数
    def read_acce_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6008].raw
        return value

    # 设置加速度系数（0-8）
    def set_acce_speed(self, node_id, a_speed):
        self.node_map[node_id].sdo[0x6008].raw = a_speed

    # 读取减速度系数
    def read_dece_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6009].raw
        return value

    # 设置减速度系数（0-8）
    def set_dece_speed(self, node_id, d_speed):
        self.node_map[node_id].sdo[0x6009].raw = d_speed

    # 读取细分数
    def read_subdivision(self, node_id):
        value = self.node_map[node_id].sdo[0x600A].raw
        return value

    # 设置细分数
    def set_subdivision(self, node_id, value):
        self.node_map[node_id].sdo[0x600A].raw = value

    # 读取最大相电流
    def read_max_phase_current(self, node_id):
        value = self.node_map[node_id].sdo[0x600B].raw
        return value

    # 设置最大相电流 (0-6000)
    def set_max_phase_current(self, node_id, value):
        self.node_map[node_id].sdo[0x600B].raw = value

    # 读取电机位置
    def read_motor_position(self, node_id):
        value = self.node_map[node_id].sdo[0x600C].raw
        return value

    # 设置电机位置
    def set_motor_position(self, node_id, value):
        self.node_map[node_id].sdo[0x600C].raw = value

    # 读取当前校准零位（绝对值编码器闭环）
    def read_calibration_zero(self, node_id):
        value = self.node_map[node_id].sdo[0x6034].raw
        return value

    # 设置校准零位
    def set_calibration_zero(self, node_id, value):
        self.node_map[node_id].sdo[0x6034].raw = value

    # 读取编码器位置（绝对值编码器闭环）
    def read_encoder_position(self, node_id):
        value = self.node_map[node_id].sdo[0x6035].raw
        return value

    # 读取电流衰减系数
    def read_current_attenuation(self, node_id):
        value = self.node_map[node_id].sdo[0x600D].raw
        return value

    # 设置电流衰减系数
    def set_current_attenuation(self, node_id, value=0):
        self.node_map[node_id].sdo[0x600D].raw = value

    # 电机使能
    def read_motor_enable(self, node_id):
        value = self.node_map[node_id].sdo[0x600E].raw
        return value

    # 电机使能设置 (0.脱机；1.使能电机)
    def set_motor_enable(self, node_id, value=1):
        self.node_map[node_id].sdo[0x600E].raw = value

    # 设置堵转后电机是否停止 开环（0.不停止；1.堵转后停止）
    def set_stall(self, node_id, value):
        self.node_map[node_id].sdo[0x601B].raw = value

    # 设置堵转检测参数 开环（范围bit bit0~6：堵转门限，有符号数；bit7~15:保留）
    def set_stall_parameter(self, node_id, value):
        self.node_map[node_id].sdo[0x6017].raw = value

    # 读取实时速度 闭环
    def read_realtime_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x6030].raw
        return value

    # 读取外部紧急停止
    def read_external_stop(self, node_id):
        value1 = self.node_map[node_id].sdo[0x600F][0x01].raw  # 外部紧急停止使能
        a = '{:08b}'.format(value1)
        value2 = self.node_map[node_id].sdo[0x600F][0x02].raw  # 外部紧急停止触发模式
        b = '{:08b}'.format(value2)
        value3 = self.node_map[node_id].sdo[0x600F][0x03].raw  # 传感器类型
        c = '{:08b}'.format(value3)
        return a, b, c

    # 设置外部紧急停止使能（bit 0禁止，1使能；bit0-紧急停止1，bit1-紧急停止2，bit4-紧急停止3）
    def set_external_stop_enable(self, node_id, value):
        self.node_map[node_id].sdo[0x600F][0x01].raw = value

    # 设置外部紧急停止触发方式（bit 0下降沿，1上升沿；bit0-停止1触发方式，bit1-停止2，bit4-停止3）
    def set_external_stop_trigger_mode(self, node_id, value):
        self.node_map[node_id].sdo[0x600F][0x02].raw = value

    # 设置传感器类型（0，1）
    # 0 上升沿触发时配置下拉电阻，下降沿触发时配置为上拉电阻，常用于PNP型传感器
    # 1 上升沿触发时配置上拉电阻，下降沿触发时配置为下拉电阻，常用于NPN型传感器
    def set_sensor_type(self, node_id, value):
        self.node_map[node_id].sdo[0x600F][0x03].raw = value

    # 设置EXT1/EXT2/EXT3稳定延时（0~200）
    def set_stable_delay(self, node_id, value):
        self.node_map[node_id].sdo[0x601a].raw = value

    # 读取通用I/O端口设置
    def read_IO_port(self, node_id):
        value1 = self.node_map[node_id].sdo[0x6011][0x01].raw  # 端口方向
        a = '{:016b}'.format(value1)
        value2 = self.node_map[node_id].sdo[0x6011][0x02].raw  # 端口配置
        b = '{:032b}'.format(value2)
        return a, b

    # 设置端口方向 （u16）
    # bit位表示，0输入，1输出；bit0~6表示GPIO1~7，bit7:EXT1,bit8:EXT2,bit9：EXT3/ENC1,bit10:ENC2,bit11:GPIO8
    def set_IO_port_direction(self, node_id, value):
        self.node_map[node_id].sdo[0x6011][0x01].raw = value

    # 设置端口配置  (范围0-0x3fffff  u32）
    # 每个端口用2bit表示，若配置为输入端口，则其值含义为:0-FLOATING,1-IPU,2-IPD,3-AIN 若配置为输出端口，则其值含义为:0-OD,1-PP
    # IO端口各位定义参见手册
    def set_IO_port_dispose(self, node_id, value):
        self.node_map[node_id].sdo[0x6011][0x02].raw = value

    # 读取通用IO端口值
    def read_IO_port_value(self, node_id):
        value = self.node_map[node_id].sdo[0x6012].raw
        a = '{:016b}'.format(value)
        return a

    # 设置端口值 (bit表示,U16，0高电平，1低电平.bit0~6:GPIO1~6,bit7:EXT1,bit8:EXT2,bit9:EXT3/ENC1,bit10:ENC2,bit11:GPIO8)
    def set_IO_port_value(self, node_id, value):
        self.node_map[node_id].sdo[0x6012].raw = value

    # 设置离线编程参数1
    # 设置离线编程指令数 (0-100)，设置离线自动运行使能 (0不能离线自动运行，1能离线自动运行）
    def set_offline_parameter1(self, node_id, number, enable):
        self.node_map[node_id].sdo[0x6018][0x01].raw = number
        self.node_map[node_id].sdo[0x6018][0x02].raw = enable

    # 设置离线编程参数2
    # 离线程序指针pointer，离线指令direct1，保持离线指令k_direct，GPIO mask,mask，运行指令direct2
    def set_offline_parameter2(self, node_id, direct1, pointer, k_direct, mask, direct2):
        self.node_map[node_id].sdo[0x6019][0x01].raw = pointer
        self.node_map[node_id].sdo[0x6019][0x02].raw = direct1
        self.node_map[node_id].sdo[0x6019][0x03].raw = k_direct
        self.node_map[node_id].sdo[0x6019][0x04].raw = mask
        self.node_map[node_id].sdo[0x6019][0x05].raw = direct2

    # 设置编码器分辨率
    def set_encoder_resolution(self, node_id, value):  # 增量式编码器闭环
        self.node_map[node_id].sdo[0x6021].raw = value

    # 读取编码器分辨率
    def read_encoder_resolution(self, node_id):  # 增量式编码器闭环
        value = self.node_map[node_id].sdo[0x6021].raw
        return value
    # def set_encoder_resolution(value=12):  #绝对值编码器闭环
    #     node.sdo[0x6021].raw = value

    # 设置KP参数(1-255)
    def set_KP(self, node_id, value):
        self.node_map[node_id].sdo[0x6023].raw = value

    # 设置KI参数（1-255）
    def set_KI(self, node_id, value):
        self.node_map[node_id].sdo[0x6024].raw = value

    # 设置KD参数(1-255)
    def set_KD(self, node_id, value=8):
        self.node_map[node_id].sdo[0x6025].raw = value

    # 设置前置滤波参数 pre_FP(1-128)
    def set_pre_FP(self, node_id, value):
        self.node_map[node_id].sdo[0x6026].raw = value

    # 设置后置滤波参数post_FP(1-255)
    def set_post_FP(self, node_id, value):
        self.node_map[node_id].sdo[0x6027].raw = value

    # 设置堵转长度参数stall_LP(1-255)
    def set_stall_LP(self, node_id, value):
        self.node_map[node_id].sdo[0x6028].raw = value

    # 力矩环使能设置(0-1)
    def set_torque_enable(self, node_id, value):
        self.node_map[node_id].sdo[0x6029].raw = value

    # 自动掉电保存使能(0-1)
    def set_power_down_save(self, node_id, value):
        self.node_map[node_id].sdo[0x2007].raw = value

    # 同步定位运动控制(同步定位速度speed，同步定位位置position)
    def set_synchronous_motion_control(self, node_id, speed, position):
        self.node_map[node_id].sdo[0x601D][0x01].raw = speed
        self.node_map[node_id].sdo[0x601D][0x02].raw = position

    # 设置pvt控制操作(0-3, 0终止pvt运动， 1启动pvt运动， 2将pvt位置速度和时间对象写入队列中， 3清楚队列中的所有pvt数据
    def set_pvt_operation(self, node_id, value):
        self.node_map[node_id].sdo[0x6010][0x01].raw = value

    # 设置pvt工作模式（0-2,0 pvt模式1，1 pvt模式2，2 pvt模式3）
    def set_pvt_workmode(self, node_id, value):
        self.node_map[node_id].sdo[0x6010][0x02].raw = value

    # 设置最大pvt点数（0-1000）
    def set_pvt_maxpoints(self, node_id, value):
        self.node_map[node_id].sdo[0x6010][0x03].raw = value

    # 设置当前pvt指针（0-1000）
    def set_pvt_pointer(self, node_id, value=0):
        self.node_map[node_id].sdo[0x6010][0x04].raw = value

    # 设置pvt模式1参数（起始索引，结束索引 0-1000）
    def set_pvt_parameter1(self, node_id, start, over):
        self.node_map[node_id].sdo[0x6010][0x05].raw = start
        self.node_map[node_id].sdo[0x6010][0x06].raw = over

    # 设置pvt模式2参数（0-1000）
    # 加速阶段起始索引ac_start,结束索引ac_over
    # 循环阶段起始索引loop_start,结束索引loop_over,循环阶段次数loop_count（0-65535）
    # 减速阶段起始索引de_start,结束索引de_over
    def set_pvt_parameter2(self, node_id, ac_start, ac_over, loop_start, loop_over, loop_count, de_start, de_over):
        self.node_map[node_id].sdo[0x6010][0x07].raw = ac_start
        self.node_map[node_id].sdo[0x6010][0x08].raw = ac_over
        self.node_map[node_id].sdo[0x6010][0x09].raw = loop_start
        self.node_map[node_id].sdo[0x6010][0x0A].raw = loop_over
        self.node_map[node_id].sdo[0x6010][0x0B].raw = loop_count
        self.node_map[node_id].sdo[0x6010][0x0C].raw = de_start
        self.node_map[node_id].sdo[0x6010][0x0D].raw = de_over

    # 设置pvt模式3参数（FIFO深度：depth，FIFO上限：upper，FIFO下限：lo_limit）（0-1000）
    def set_pvt_parameter3(self, node_id, depth, upper, lo_limit):
        self.node_map[node_id].sdo[0x6010][0x0E].raw = depth
        self.node_map[node_id].sdo[0x6010][0x0F].raw = lo_limit
        self.node_map[node_id].sdo[0x6010][0x10].raw = upper

    # 设置pvt位置
    def set_pvt_position(self, node_id, value):
        self.node_map[node_id].sdo[0x6010][0x11].raw = value

    # 设置pvt速度
    def set_pvt_speed(self, node_id, value):
        self.node_map[node_id].sdo[0x6010][0x12].raw = value

    # 设置pvt时间
    def set_pvt_time(self, node_id, value=0):
        self.node_map[node_id].sdo[0x6010][0x13].raw = value

    # 设置pp模式参数1（范围： >150）
    # 加速度ac_speed,减速度de_speed,启动速度start_speed,停止速度stop_speed
    def set_pp_acspeed(self, node_id, ac_speed):
        self.node_map[node_id].sdo[0x602d][0x01].raw = ac_speed

    def set_pp_despeed(self, node_id, de_speed):
        self.node_map[node_id].sdo[0x602d][0x02].raw = de_speed

    def set_pp_start_speed(self, node_id, start_speed):
        self.node_map[node_id].sdo[0x602d][0x03].raw = start_speed

    def set_pp_stop_speed(self, node_id, stop_speed):
        self.node_map[node_id].sdo[0x602d][0x04].raw = stop_speed

    def read_pp_ac_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x602d][0x01].raw
        return value

    def read_pp_de_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x602d][0x02].raw
        return value

    def read_pp_start_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x602d][0x03].raw
        return value

    def read_pp_stop_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x602d][0x04].raw
        return value

    # 设置pp模式参数2
    # 控制字control_word，状态字Status_word(0-0xFFFF)
    def set_pp_control_word(self, node_id, control_word):
        self.node_map[node_id].sdo[0x602e][0x01].raw = control_word

    def set_pp_Status_word(self, node_id, Status_word):
        self.node_map[node_id].sdo[0x602e][0x02].raw = Status_word

    # 读取pp模式运行速度
    def read_pp_speed(self, node_id):
        value = self.node_map[node_id].sdo[0x602e][0x03].raw
        return value

    # pv模式启动
    def pp_start(self, node_id):
        self.node_map[node_id].sdo[0x6005].raw = 5
        self.node_map[node_id].sdo[0x602e][0x01].raw = 0

    # pv模式停止
    def pp_stop(self, node_id):
        self.node_map[node_id].sdo[0x602e][0x01].raw = 256

    # 运行速度speed,目标位置target_location
    def set_pp_speed(self, node_id, speed):
        self.node_map[node_id].sdo[0x602e][0x03].raw = speed

    def set_pp_step(self, node_id, step):
        self.node_map[node_id].sdo[0x602e][0x04].raw = step

    # pv模式模拟量定位设置
    # 模拟量定位使能enable(0关闭，1打开)，模拟量起始ad码ad（0-4096），
    # 模拟量调节间隔clearance（0-65535），模拟量调节触发值trigger_value（0-65535）
    # 模拟量位置最小值min_value,模拟量位置最大值max_value
    def set_pv_analog_location(self, node_id, enable, ad, clearance, trigger_value, min_value, max_value):
        self.node_map[node_id].sdo[0x602f][0x01].raw = enable
        self.node_map[node_id].sdo[0x602f][0x02].raw = ad
        self.node_map[node_id].sdo[0x602f][0x03].raw = clearance
        self.node_map[node_id].sdo[0x602f][0x04].raw = trigger_value
        self.node_map[node_id].sdo[0x602f][0x05].raw = min_value
        self.node_map[node_id].sdo[0x602f][0x06].raw = max_value

    # 刹车控制（0-100）
    def set_brake_control(self, node_id, value):
        self.node_map[node_id].sdo[0x6016].raw = value

    # 模拟量输入（0-4095）
    def set_pv_analog_input(self, node_id, value):
        self.node_map[node_id].sdo[0x602B].raw = value

    # 步进通知设置（状态state，位置position1，位置position2）
    # 步进通知状态bit表示，0输入，1输出，bit0位置点1有效，bit1位置点2有效
    def set_pv_stepping_notice(self, node_id, state, position1, position2):
        self.node_map[node_id].sdo[0x602c][0x01].raw = state
        self.node_map[node_id].sdo[0x602c][0x02].raw = position1
        self.node_map[node_id].sdo[0x602c][0x03].raw = position2

    # 掉电行为 (掉电行为控制字control_word，脱机电压offline_v（0-65535），抱闸电压Hold_brake_v（0-65535）)
    def set_pv_power_down(self, node_id, control_word, offline_v, hold_brake_v):
        self.node_map[node_id].sdo[0x6031][0x01].raw = control_word
        self.node_map[node_id].sdo[0x6031][0x02].raw = offline_v
        self.node_map[node_id].sdo[0x6031][0x03].raw = hold_brake_v

    # 心跳包
    def set_heartbeat(self, node_id, value=0):
        self.node_map[node_id].sdo[0x1017].raw = value

    def set_pdo_mapping(self, node_id, pdo_num, inhibit_time, event_timer, vars, callback):
        self.node_map[node_id].tpdo[pdo_num].cob_id = 0x180 + node_id
        self.node_map[node_id].tpdo[pdo_num].trans_type = 255
        self.node_map[node_id].tpdo[pdo_num].inhibit_time = inhibit_time
        self.node_map[node_id].tpdo[pdo_num].event_timer = event_timer
        self.node_map[node_id].tpdo[pdo_num].enabled = True
        self.node_map[node_id].tpdo[pdo_num].clear()
        for var in vars:
            self.node_map[node_id].tpdo[pdo_num].add_variable(var)

        self.node_map[node_id].tpdo[pdo_num].add_callback(callback)
        self.node_map[node_id].tpdo[pdo_num].save()

    def close_tpdo(self, node_id, pdo_num):
        self.node_map[node_id].tpdo[pdo_num].trans_type = 255
        self.node_map[node_id].tpdo[pdo_num].inhibit_time = 0
        self.node_map[node_id].tpdo[pdo_num].event_timer = 0
        self.node_map[node_id].tpdo[pdo_num].enabled = False
        self.node_map[node_id].tpdo[pdo_num].clear()
        self.node_map[node_id].tpdo[pdo_num].save()

