import time
import can
from canopen import Network
from zlgcan import *
import ctypes


class ZLGCanNetBus(can.BusABC):
    """
    适配 ZLG CANNET-TCP 的 python-can 接口实现
    """

    def __init__(self, ip, port, device_type=ZCAN_CANETTCP, device_index=0, channel=0, **kwargs):
        self.channel_info = f"ZLG-NET-{ip}:{port}"
        # 1. 初始化 ZLG API
        self.zcanlib = ZCAN()
        self.device_handle = self.zcanlib.OpenDevice(device_type, device_index, 0)

        if self.device_handle == INVALID_DEVICE_HANDLE:
            raise can.CanError("打开设备失败！请检查驱动或库路径。")

        # 2. 配置参数 (IP, Port, Work Mode)
        # 默认为客户端模式(0)
        if self.zcanlib.ZCAN_SetValue(self.device_handle, "0/work_mode", "0".encode("utf-8")) == ZCAN_STATUS_ERR:
            raise can.CanError("设置工作模式失败")

        if self.zcanlib.ZCAN_SetValue(self.device_handle, "0/work_port", port.encode("utf-8")) == ZCAN_STATUS_ERR:
            raise can.CanError("设置目标端口失败")
        if self.zcanlib.ZCAN_SetValue(self.device_handle, "0/ip", ip.encode("utf-8")) == ZCAN_STATUS_ERR:
            raise can.CanError("设置目标IP失败")

        # 3. 初始化通道
        chn_init_cfg = ZCAN_CHANNEL_INIT_CONFIG()
        memset(addressof(chn_init_cfg), 0, sizeof(chn_init_cfg))
        # 根据实际需求配置波特率等，CANNET通常在网页端配置，这里初始化主要建立连接

        self.chn_handle = self.zcanlib.InitCAN(self.device_handle, channel, chn_init_cfg)
        if self.chn_handle is None or self.chn_handle == 0:
            raise can.CanError("启动通道失败")

        # 4. 启动 CAN
        if self.zcanlib.StartCAN(self.chn_handle) == ZCAN_STATUS_ERR:
            raise can.CanError("StartCAN 失败，请检查网络连接")

        # 调用父类初始化
        super().__init__(channel=channel, **kwargs)

    def send(self, msg, timeout=None):
        """
        发送 CAN 帧
        """
        transmit_num = 1
        z_msg = (ZCAN_Transmit_Data * transmit_num)()
        memset(ctypes.addressof(z_msg), 0, sizeof(z_msg))

        # 填充 ID 和 标志位
        # ZLG API 通常在 ID 的高位标记 扩展帧(bit 31) 和 远程帧(bit 30)
        can_id = msg.arbitration_id
        if msg.is_extended_id:
            can_id |= 0x80000000
        if msg.is_remote_frame:
            can_id |= 0x40000000

        z_msg[0].frame.can_id = can_id
        z_msg[0].frame.can_dlc = msg.dlc
        z_msg[0].transmit_type = 0  # 0-正常发送, 2-自发自收

        # 填充数据
        for i in range(msg.dlc):
            z_msg[0].frame.data[i] = msg.data[i]

        ret = self.zcanlib.Transmit(self.chn_handle, z_msg, transmit_num)
        if ret != transmit_num:
            raise can.CanOperationError("发送失败")

    def _recv_internal(self, timeout):
        """
        接收 CAN 帧 (被 python-can 内部循环调用)
        timeout: 单位秒
        """
        # ZLG Receive 的 timeout 单位通常是毫秒
        wait_time_ms = int(timeout * 1000) if timeout else 0

        # 获取缓冲区帧数 (非阻塞检查)
        rcv_num = self.zcanlib.GetReceiveNum(self.chn_handle, ZCAN_TYPE_CAN)

        if rcv_num == 0:
            # 如果没有数据且需要等待，可以使用 sleep 或带超时的 Receive
            # 注意：如果 ZCAN Receive 是阻塞的，可以直接调用。
            # 这里为了简单，如果此时没数据且有 timeout，简单 sleep 一下避免 CPU 100%
            if timeout and timeout > 0:
                time.sleep(min(timeout, 0.01))
            return None, False

        # 读取 1 帧
        rcv_msg, num = self.zcanlib.Receive(self.chn_handle, 1, wait_time_ms)

        if num > 0:
            frame = rcv_msg[0].frame

            # 解析 ID 和 标志位
            raw_id = frame.can_id
            is_extended = bool(raw_id & 0x80000000)
            is_remote = bool(raw_id & 0x40000000)
            arbitration_id = raw_id & 0x1FFFFFFF

            # 提取数据
            data = bytearray(frame.data[:frame.can_dlc])

            # 构建 python-can 的 Message 对象
            msg = can.Message(
                arbitration_id=arbitration_id,
                data=data,
                dlc=frame.can_dlc,
                is_extended_id=is_extended,
                is_remote_frame=is_remote,
                timestamp=rcv_msg[0].timestamp / 1000000.0  # 假设是微秒，视具体 ZLG 版本而定
            )
            return msg, False

        return None, False

    def shutdown(self):
        """
        关闭连接
        """
        self.zcanlib.CloseDevice(self.device_handle)